// script.js (logic cho trang chủ)

document.addEventListener('DOMContentLoaded', () => {
    const productListDiv = document.getElementById('productList');
    const cartItemCountSpan = document.getElementById('cartItemCount');
    const toastContainer = document.getElementById('toastContainer');

    // Lấy tham chiếu đến ô input tìm kiếm và nút tìm kiếm (với class mới)
    const searchInput = document.querySelector('.search-bar input[type="text"]');
    const searchButton = document.querySelector('.search-bar .search-button-text'); // Cập nhật selector này

    function showToast(message, duration = 3000) {
        const toast = document.createElement('div');
        toast.classList.add('toast-message');
        toast.textContent = message;
        toastContainer.appendChild(toast);
        setTimeout(() => {
            toast.remove();
        }, duration + 500);
    }

    function getCartFromLocalStorage() {
        const cartString = localStorage.getItem('cart');
        return cartString ? JSON.parse(cartString) : [];
    }

    function saveCartToLocalStorage(cart) {
        localStorage.setItem('cart', JSON.stringify(cart));
    }

    function updateCartItemCount() {
        const cart = getCartFromLocalStorage();
        const totalItems = cart.reduce((sum, item) => sum + item.quantity, 0);
        cartItemCountSpan.textContent = totalItems;
    }

    function addToCart(productId, color, size, quantity = 1) {
        // Đọc tất cả sản phẩm từ localStorage
        const allProducts = JSON.parse(localStorage.getItem('allProducts') || '[]');
        const product = allProducts.find(p => p.id === productId);

        if (product) {
            let cart = getCartFromLocalStorage();
            const uniqueItemId = `${productId}-${color || 'no-color'}-${size || 'no-size'}`;
            const existingItemIndex = cart.findIndex(item => item.uniqueId === uniqueItemId);

            if (existingItemIndex > -1) {
                cart[existingItemIndex].quantity += quantity;
            } else {
                cart.push({
                    uniqueId: uniqueItemId,
                    id: product.id,
                    name: product.name,
                    price: product.price,
                    image: product.image, // Ảnh chính của sản phẩm
                    color: color,
                    size: size,
                    quantity: quantity
                });
            }
            saveCartToLocalStorage(cart);
            updateCartItemCount();
            showToast(`Đã thêm "${product.name} (${color || ''} - ${size || ''})" vào giỏ hàng!`);
        }
    }

    // Hàm hiển thị danh sách sản phẩm, có thêm tham số 'searchTerm' để lọc
    function displayProducts(searchTerm = '') {
        // Luôn đọc tất cả sản phẩm từ localStorage
        const allProducts = JSON.parse(localStorage.getItem('allProducts') || '[]');
        let productsToDisplay = allProducts; // Mặc định hiển thị tất cả

        // Nếu có từ khóa tìm kiếm, lọc sản phẩm
        if (searchTerm) {
            const lowerCaseSearchTerm = searchTerm.toLowerCase();
            productsToDisplay = allProducts.filter(product =>
                product.name.toLowerCase().includes(lowerCaseSearchTerm) ||
                product.description.toLowerCase().includes(lowerCaseSearchTerm)
            );
        }

        if (!productListDiv) return; // Đảm bảo element tồn tại
        productListDiv.innerHTML = ''; // Xóa nội dung cũ

        // Hiển thị thông báo nếu không tìm thấy sản phẩm nào
        if (productsToDisplay.length === 0) {
            productListDiv.innerHTML = '<p>Không tìm thấy sản phẩm nào phù hợp.</p>';
            return;
        }

        productsToDisplay.forEach(product => {
            const productItem = document.createElement('div');
            productItem.classList.add('product-item');
            productItem.innerHTML = `
                <div class="product-image-wrapper">
                    <img src="${product.image}" alt="${product.name}">
                </div>
                <div class="product-info">
                    <h3>${product.name}</h3>
                    <p class="price">${product.price.toLocaleString('th-TH', { style: 'currency', currency: 'THB' })}</p>
                    <button class="add-to-cart-btn-home" data-id="${product.id}">ซื้อสินค้า</button>
                </div>
            `;
            productListDiv.appendChild(productItem);
        });

        // Gán sự kiện click cho các nút "Thêm vào giỏ" trên trang chủ
        document.querySelectorAll('.add-to-cart-btn-home').forEach(button => {
            button.addEventListener('click', (e) => {
                e.stopPropagation(); // Ngăn sự kiện click lan ra phần tử cha (product-item)
                const productId = parseInt(e.target.dataset.id);
                // Quan trọng: Luôn tìm sản phẩm từ danh sách `allProducts` gốc
                const product = allProducts.find(p => p.id === productId);

                if (product) {
                    let defaultColor = 'No Color';
                    let defaultSize = 'Freesize';

                    if (product.variants && product.variants.length > 0) {
                        defaultColor = product.variants[0].color;
                        defaultSize = product.variants[0].sizes && product.variants[0].sizes.length > 0 ? product.variants[0].sizes[0] : 'Freesize';
                    }
                    addToCart(product.id, defaultColor, defaultSize);
                }
            });
        });

        // Thêm sự kiện click cho toàn bộ product-item để chuyển hướng sang trang chi tiết
        document.querySelectorAll('.product-item').forEach(item => {
            item.addEventListener('click', (e) => {
                // Đảm bảo không kích hoạt khi click vào nút "Thêm vào giỏ"
                if (e.target.classList.contains('add-to-cart-btn-home')) {
                    return;
                }
                const productId = parseInt(item.querySelector('.add-to-cart-btn-home').dataset.id);
                window.location.href = `product-detail.html?id=${productId}`;
            });
        });
    }

    // --- Xử lý chức năng tìm kiếm ---
    function handleSearch() {
        const searchTerm = searchInput.value.trim(); // Lấy giá trị và xóa khoảng White thừa
        displayProducts(searchTerm); // Gọi lại hàm hiển thị sản phẩm với từ khóa tìm kiếm
    }

    // Gắn sự kiện click cho nút tìm kiếm
    if (searchButton) { // Kiểm tra để đảm bảo nút tồn tại
        searchButton.addEventListener('click', handleSearch);
    }

    // Gắn sự kiện 'keypress' để người dùng có thể nhấn Enter để tìm kiếm
    if (searchInput) { // Kiểm tra để đảm bảo input tồn tại
        searchInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                handleSearch();
            }
        });
    }
    // --- Kết thúc xử lý chức năng tìm kiếm ---

    // Khởi tạo hiển thị khi tải trang (hiển thị tất cả sản phẩm ban đầu)
    displayProducts();
    updateCartItemCount(); // Cập nhật số lượng giỏ hàng khi tải trang
});